package cn.tsign.hz.run;

import cn.tsign.hz.constant.ConfigConstant;
import cn.tsign.hz.core.ClientHelper;
import cn.tsign.hz.core.FileHelper;
import cn.tsign.hz.core.SignHelper;
import cn.tsign.hz.exception.DefineException;
import com.timevale.esign.paas.tech.bean.bean.PosBean;
import com.timevale.esign.paas.tech.bean.request.OrgSignParam;
import com.timevale.esign.paas.tech.bean.request.SignFilePdfParam;
import com.timevale.esign.paas.tech.client.ServiceClient;
import com.timevale.esign.paas.tech.enums.SignType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

/***
 * description: 企业用户签
 */

public class RunOrgSign {

    private static final Logger LOGGER = LoggerFactory.getLogger(RunOrgSign.class);
    private static SignHelper signHelper;

    static {
        try {
            if (true) {
                //1、注册客户端，全局使用，只需注册一次
                ClientHelper.registClient();
            }

            //2、获取已初始化的客户端，以便后续正常调用SDK提供的各种服务，全局使用，只需获取一次
            ServiceClient serviceClient = ClientHelper.getServiceClient(ConfigConstant.PROJECT_ID);

            //3、实例化辅助类
            signHelper = new SignHelper(serviceClient);
        } catch (DefineException e) {
            e.getE().printStackTrace();
        }
    }

//--------------------------------公有方法 start-------------------------------------

    public static void main(String[] args) throws DefineException {

        switch (0) {
            case 0:
                LOGGER.info("====>场景演示：企业用户PDF文件签署（文件路径方式）<=====");
                orgSignByPdfFile();
                break;
            case 1:
                LOGGER.info("====>场景演示：企业用户PDF文件签署（文件流方式）<=====");
                orgSignByPdfStream();
                break;
            default:
                LOGGER.info("====>请选择应用场景<=====");
                break;
        }
    }
//--------------------------------公有方法 end---------------------------------------

//--------------------------------私有方法 start-------------------------------------

    // 当前程序所在文件目录
    private static final String ROOT_FOLDER = new File("").getAbsolutePath();
    //文件地址前缀拼接（可根据实际场景自定义）
    private static final String PATH_PREFEX = ROOT_FOLDER + File.separator + "pdf" + File.separator;

    /**
     * 文件路径方式企业用户PDF文件签署
     */
    private static void orgSignByPdfFile() throws DefineException {
        /**
         * 企业用户PDF文件路径签署
         */
        // 签署文件信息
        SignFilePdfParam file = new SignFilePdfParam();
        file.setSrcPdfFile(PATH_PREFEX + "test.pdf");// 待签署PDF文件本地路径，含文件名（与bytes至少有一个不为空）
        file.setDstPdfFile(PATH_PREFEX + "Signed_Organize.pdf");// 签署后PDF文件本地路径，含文件名（为空时返回签署后的文件流）
        file.setFileName("我的企业签署文件.pdf");

        // 签章位置信息
        List<PosBean> posBeans = new ArrayList<>();
        PosBean signPos = new PosBean();
        signPos.setPosPage("1");// 签署页码（除关键字签章外，页码均不可为空）若为多页签章，支持指定多个页码，例如格式：“1-3,5,8“
        signPos.setPosX(150);// 签署位置X坐标，若为关键字定位，相对于关键字的X坐标偏移量，默认0
        signPos.setPosY(0);// 签署位置Y坐标，若为关键字定位，相对于关键字的Y坐标偏移量，默认0
        signPos.setKeyWord("甲方盖章");// 关键字，仅限关键字签章时有效，若为关键字定位时，不可空
        signPos.setWidth(159);// 印章展现宽度，将以此宽度对印章图片做同比缩放。
        signPos.setAddSignTime(true);// 是否显示本地签署时间，需要width设置92以上才可以看到时间
        posBeans.add(signPos);

        //企业用户印章Base64数据，需要和企业账号对应的企业名字一致
        String sealData = "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";

        //传入企业用户签署参数内
        OrgSignParam orgSignParam = new OrgSignParam();
        /**！！！注意这里企业如果是授权给平台的，只需要传企业账号：accountId即可签署*/
        /**！！！注意这里企业如果是授权给经办人的，那么即需要传企业账号：accountId，还需要传经办人账号：willingnessAccountId和经办人个人认证流程：willingnessId*/
        orgSignParam.setAccountId("34C3*******26589B9575");//企业账号ID（创建企业签署账号接口返回），需要提前授权给平台
        orgSignParam.setWillingnessAccountId("");//经办人个人账号ID（创建个人签署账号接口返回）
        orgSignParam.setWillingnessId("");//经办人个人认证流程ID（通过身份核验认证服务-个人认证模块选择任意一种方式认证完成）
        orgSignParam.setSealData(sealData);//印章Base64
        orgSignParam.setPosBeans(posBeans);//签章位置信息
        orgSignParam.setSignType(SignType.Key);//签章类型：Single，单页签章； Multi，多页签章； Edges，签骑缝章； Key，关键字签章
        orgSignParam.setFileBean(file);//签署文件信息
        signHelper.orgSign(orgSignParam);

    }


    /**
     * 文件流方式企业用户PDF文件签署
     */
    private static void orgSignByPdfStream() throws DefineException {

        /**
         * 企业用户PDF文件流签署
         */
        // 签署文件信息
        SignFilePdfParam file = new SignFilePdfParam();
        //PDF文件信息
        String srcPdfPath = PATH_PREFEX + "test.pdf";
        //获取PDF文件的字节流
        byte[] srcPdfBytes = FileHelper.getFileBytes(srcPdfPath);
        file.setStreamFile(srcPdfBytes);// 待填充PDF文件本地二进制数据
        //file.setDstPdfFile(PATH_PREFEX + "Signed_Platform.pdf");// 签署后PDF文件本地路径，含文件名（为空时返回签署后的文件流）
        file.setFileName("我的企业签署文件.pdf");

        // 签章位置信息
        List<PosBean> posBeans = new ArrayList<>();
        PosBean signPos = new PosBean();
        signPos.setPosPage("1");// 签署页码（除关键字签章外，页码均不可为空）若为多页签章，支持指定多个页码，例如格式：“1-3,5,8“
        signPos.setPosX(150);// 签署位置X坐标，若为关键字定位，相对于关键字的X坐标偏移量，默认0
        signPos.setPosY(0);// 签署位置Y坐标，若为关键字定位，相对于关键字的Y坐标偏移量，默认0
        signPos.setKeyWord("甲方盖章");// 关键字，仅限关键字签章时有效，若为关键字定位时，不可空
        signPos.setWidth(159);// 印章展现宽度，将以此宽度对印章图片做同比缩放。
        signPos.setAddSignTime(true);// 是否显示本地签署时间，需要width设置92以上才可以看到时间
        posBeans.add(signPos);

        //印章数据二选一传入
        //企业用户印章Base64数据，需要和企业账号对应的企业名字一致
        String sealData = "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";

        //传入企业用户签署参数内
        OrgSignParam orgSignParam = new OrgSignParam();
        /**！！！注意这里企业如果是授权给平台的，只需要传企业账号：accountId即可签署*/
        /**！！！注意这里企业如果是授权给经办人的，那么即需要传企业账号：accountId，还需要传经办人账号：willingnessAccountId和经办人个人认证流程：willingnessId*/
        orgSignParam.setAccountId("34C37*********C26589B9575");//企业账号ID（创建企业签署账号接口返回），需要提前授权给平台
        orgSignParam.setWillingnessAccountId("");//经办人个人账号ID（创建个人签署账号接口返回）
        orgSignParam.setWillingnessId("");//经办人个人认证流程ID（通过身份核验认证服务-个人认证模块选择任意一种方式认证完成）
        orgSignParam.setSealData(sealData);//印章Base64
        orgSignParam.setPosBeans(posBeans);//签章位置信息
        orgSignParam.setSignType(SignType.Key);//签章类型：Single，单页签章； Multi，多页签章； Edges，签骑缝章； Key，关键字签章
        orgSignParam.setFileBean(file);//签署文件信息
        signHelper.orgSign(orgSignParam);


    }

//--------------------------------私有方法 end---------------------------------------
}

