package cn.tsign.hz.run;

import cn.tsign.hz.constant.ConfigConstant;
import cn.tsign.hz.core.ClientHelper;
import cn.tsign.hz.core.FileHelper;
import cn.tsign.hz.core.SignHelper;
import cn.tsign.hz.exception.DefineException;
import com.timevale.esign.paas.tech.bean.bean.PosBean;
import com.timevale.esign.paas.tech.bean.request.PlatformSignParam;
import com.timevale.esign.paas.tech.bean.request.SignFilePdfParam;
import com.timevale.esign.paas.tech.client.ServiceClient;
import com.timevale.esign.paas.tech.enums.SealSpecEnum;
import com.timevale.esign.paas.tech.enums.SignType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.ArrayList;
import java.util.List;


/***
 * description: 平台自身签
 */

public class RunPlatformSign {

    private static final Logger LOGGER = LoggerFactory.getLogger(RunPlatformSign.class);
    private static SignHelper signHelper;

    static {
        try {
            if (true) {
                //1、注册客户端，全局使用，只需注册一次
                ClientHelper.registClient();
            }

            //2、获取已初始化的客户端，以便后续正常调用SDK提供的各种服务，全局使用，只需获取一次
            ServiceClient serviceClient = ClientHelper.getServiceClient(ConfigConstant.PROJECT_ID);

            //3、实例化辅助类
            signHelper = new SignHelper(serviceClient);
        } catch (DefineException e) {
            e.getE().printStackTrace();
        }
    }

//--------------------------------公有方法 start-------------------------------------

    public static void main(String[] args) throws DefineException {

        switch (0) {
            case 0:
                LOGGER.info("====>场景演示：平台自身自动落章（文件路径方式）<=====");
                platformSignByPdfFile();
                break;
            case 1:
                LOGGER.info("====>场景演示：平台自身自动落章（文件流方式）<=====");
                platformSignByPdfStream();
                break;
            default:
                LOGGER.info("====>请选择应用场景<=====");
                break;
        }
    }
//--------------------------------公有方法 end---------------------------------------

//--------------------------------私有方法 start-------------------------------------

    // 当前程序所在文件目录
    private static final String ROOT_FOLDER = new File("").getAbsolutePath();
    //文件地址前缀拼接（可根据实际场景自定义）
    private static final String PATH_PREFEX = ROOT_FOLDER + File.separator + "pdf" + File.separator;

    /**
     * 文件路径方式平台自身自动落章
     */
    private static void platformSignByPdfFile() throws DefineException {
        /**
         * 平台自身PDF文件路径签署
         */
        // 签署文件信息
        SignFilePdfParam file = new SignFilePdfParam();
        file.setSrcPdfFile(PATH_PREFEX + "软件销售new.pdf");// 待签署PDF文件本地路径，含文件名（与bytes至少有一个不为空）
        file.setDstPdfFile(PATH_PREFEX + "Signed_Platform.pdf");// 签署后PDF文件本地路径，含文件名（为空时返回签署后的文件流）
        file.setFileName("我的平台自身签署文件.pdf");

        // 签章位置信息
        List<PosBean> posBeans = new ArrayList<>();
        PosBean signPos = new PosBean();
        signPos.setPosPage("2");// 签署页码（除关键字签章外，页码均不可为空）若为多页签章，支持指定多个页码，例如格式：“1-3,5,8“
        signPos.setPosX(233);// 签署位置X坐标，若为关键字定位，相对于关键字的X坐标偏移量，默认0
        signPos.setPosY(401);// 签署位置Y坐标，若为关键字定位，相对于关键字的Y坐标偏移量，默认0
        signPos.setKeyWord("甲方");// 关键字，仅限关键字签章时有效，若为关键字定位时，不可空
        signPos.setWidth(159);// 印章展现宽度，将以此宽度对印章图片做同比缩放。
        signPos.setAddSignTime(false);// 是否显示本地签署时间，需要width设置92以上才可以看到时间
        posBeans.add(signPos);

        //印章数据二选一传入
        //平台方自身企业的电子印章结构数据，需要和平台企业的名字一致
        //String sealData = "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";
        String sealId = "c3789282******-b3dc-337d9582def7";//印章也可以直接传入平台在e签宝官网的印章编号

        //传入企业用户签署参数内
        PlatformSignParam platformSignParam = new PlatformSignParam();
        //platformSignParam.setSealData(sealData);//印章Base64
        platformSignParam.setSealId(sealId);//印章ID数据
        platformSignParam.setPosBeans(posBeans);//签章位置信息
        platformSignParam.setSignType(SignType.Single);//签章类型：Single，单页签章； Multi，多页签章； Edges，签骑缝章； Key，关键字签章
        platformSignParam.setFileBean(file);//签署文件信息
        platformSignParam.setSealSpec(SealSpecEnum.GB_SEAL);//印章规格，默认:：IMAGE；IMAGE-图片base64，GB_SEAL-国标印章
        signHelper.platformSign(platformSignParam);

    }


    /**
     * 文件流方式平台自身自动落章
     */
    private static void platformSignByPdfStream() throws DefineException {

        /**
         * 平台自身PDF文件流签署
         */
        // 签署文件信息
        SignFilePdfParam file = new SignFilePdfParam();
        //PDF文件信息
        String srcPdfPath = PATH_PREFEX + "test.pdf";
        //获取PDF文件的字节流
        byte[] srcPdfBytes = FileHelper.getFileBytes(srcPdfPath);
        file.setStreamFile(srcPdfBytes);// 待填充PDF文件本地二进制数据
        file.setFileName("我的平台自身签署文件.pdf");
        //file.setDstPdfFile(PATH_PREFEX + "Signed_Platform.pdf");// 签署后PDF文件本地路径，含文件名（为空时返回签署后的文件流）

        // 签章位置信息
        List<PosBean> posBeans = new ArrayList<>();
        PosBean signPos = new PosBean();
        signPos.setPosPage("1");// 签署页码（除关键字签章外，页码均不可为空）若为多页签章，支持指定多个页码，例如格式：“1-3,5,8“
        signPos.setPosX(150);// 签署位置X坐标，若为关键字定位，相对于关键字的X坐标偏移量，默认0
        signPos.setPosY(0);// 签署位置Y坐标，若为关键字定位，相对于关键字的Y坐标偏移量，默认0
        signPos.setKeyWord("甲方盖章");// 关键字，仅限关键字签章时有效，若为关键字定位时，不可空
        signPos.setWidth(159);// 印章展现宽度，将以此宽度对印章图片做同比缩放。
        signPos.setAddSignTime(true);// 是否显示本地签署时间，需要width设置92以上才可以看到时间
        posBeans.add(signPos);

        //印章数据二选一传入
        //印章Base64数据，需要和调用方企业（应用ID所属的企业）对应的主体名称一致
        String sealData = "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";
        //String sealId = "985ec1e1-1111-0000-2222-123456789"//印章也可以直接传入平台在e签宝官网的印章编号

        //传入平台自身签署参数内
        PlatformSignParam platformSignParam = new PlatformSignParam();
        platformSignParam.setSealData(sealData);//印章Base64
        //platformSignParam.setSealId(sealId);//印章ID数据
        platformSignParam.setPosBeans(posBeans);//签章位置信息
        platformSignParam.setSignType(SignType.Key);//签章类型：Single，单页签章； Multi，多页签章； Edges，签骑缝章； Key，关键字签章
        platformSignParam.setFileBean(file);//签署文件信息
        signHelper.platformSign(platformSignParam);


    }

//--------------------------------私有方法 end---------------------------------------
}

